"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const db_1 = require("@b/db");
const sequelize_1 = require("sequelize");
const query_1 = require("@b/utils/query");
const error_1 = require("@b/utils/error");
const utils_1 = require("./utils");
exports.metadata = {
    summary: "Verifies a 2Checkout payment",
    description: "Verifies a 2Checkout payment using the order reference and updates the transaction status accordingly",
    operationId: "verify2CheckoutPayment",
    tags: ["Finance", "Deposit"],
    logModule: "2CHECKOUT_DEPOSIT",
    logTitle: "Verify 2Checkout payment",
    requestBody: {
        required: true,
        content: {
            "application/json": {
                schema: {
                    type: "object",
                    properties: {
                        orderReference: {
                            type: "string",
                            description: "2Checkout order reference",
                        },
                        refNo: {
                            type: "string",
                            description: "2Checkout reference number",
                        },
                        signature: {
                            type: "string",
                            description: "2Checkout signature for verification",
                        },
                        status: {
                            type: "string",
                            description: "Payment status from 2Checkout",
                        },
                    },
                    required: ["orderReference", "refNo"],
                },
            },
        },
    },
    responses: {
        200: {
            description: "Payment verification completed successfully",
            content: {
                "application/json": {
                    schema: {
                        type: "object",
                        properties: {
                            success: { type: "boolean" },
                            transaction: { type: "object" },
                            message: { type: "string" },
                        },
                    },
                },
            },
        },
        401: query_1.unauthorizedResponse,
        404: (0, query_1.notFoundMetadataResponse)("Transaction"),
        500: query_1.serverErrorResponse,
    },
    requiresAuth: true,
};
exports.default = async (data) => {
    const { user, body, ctx } = data;
    if (!(user === null || user === void 0 ? void 0 : user.id)) {
        ctx === null || ctx === void 0 ? void 0 : ctx.fail("User not authenticated");
        throw (0, error_1.createError)({ statusCode: 401, message: "Unauthorized" });
    }
    const { orderReference, refNo, signature, status } = body;
    try {
        const config = (0, utils_1.use2Checkout)();
        // Find the transaction by order reference
        const transaction = await db_1.models.transaction.findOne({
            where: {
                userId: user.id,
                description: {
                    [sequelize_1.Op.like]: `%${orderReference}%`,
                },
                status: "PENDING",
            },
            include: [
                {
                    model: db_1.models.wallet,
                    as: "wallet",
                },
            ],
        });
        if (!transaction) {
            throw (0, error_1.createError)({
                statusCode: 404,
                message: "Transaction not found or already processed",
            });
        }
        // Verify signature if provided
        if (signature) {
            const verificationParams = {
                orderReference,
                refNo,
                status: status || "COMPLETE",
            };
            const isValidSignature = (0, utils_1.verify2CheckoutSignature)(verificationParams, signature, config.secretKey);
            if (!isValidSignature) {
                throw (0, error_1.createError)({
                    statusCode: 400,
                    message: "Invalid signature",
                });
            }
        }
        // Update transaction based on status
        const isSuccessful = status === "COMPLETE" || status === "AUTHRECEIVED";
        await db_1.sequelize.transaction(async (t) => {
            if (isSuccessful) {
                // Update transaction status to completed
                await transaction.update({
                    status: "COMPLETED",
                    metadata: JSON.stringify({
                        ...JSON.parse(transaction.metadata || "{}"),
                        refNo,
                        orderReference,
                        gateway: "2checkout",
                        verifiedAt: new Date().toISOString(),
                    }),
                }, { transaction: t });
                // Update wallet balance
                const wallet = transaction.wallet;
                if (wallet) {
                    const newBalance = parseFloat(wallet.balance) + parseFloat(transaction.amount) - parseFloat(transaction.fee || "0");
                    await wallet.update({ balance: newBalance }, { transaction: t });
                }
            }
            else {
                // Update transaction status to failed
                await transaction.update({
                    status: "FAILED",
                    metadata: JSON.stringify({
                        ...JSON.parse(transaction.metadata || "{}"),
                        refNo,
                        orderReference,
                        gateway: "2checkout",
                        failureReason: status,
                        verifiedAt: new Date().toISOString(),
                    }),
                }, { transaction: t });
            }
        });
        return {
            success: isSuccessful,
            transaction: await transaction.reload(),
            message: isSuccessful
                ? "Payment verified and processed successfully"
                : `Payment failed with status: ${status}`,
        };
    }
    catch (error) {
        if (error.statusCode) {
            throw error;
        }
        throw (0, error_1.createError)({
            statusCode: 500,
            message: `Error verifying 2Checkout payment: ${error.message}`,
        });
    }
};
