"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.metadata = void 0;
const error_1 = require("@b/utils/error");
const token_1 = require("@b/utils/token");
const db_1 = require("@b/db");
const passwords_1 = require("@b/utils/passwords");
const utils_1 = require("../utils");
exports.metadata = {
    summary: "Verifies a password reset token and sets the new password",
    operationId: "verifyPasswordReset",
    tags: ["Auth"],
    description: "Verifies a password reset token and sets the new password",
    requiresAuth: false,
    logModule: "PASSWORD",
    logTitle: "Password reset confirmation",
    requestBody: {
        required: true,
        content: {
            "application/json": {
                schema: {
                    type: "object",
                    properties: {
                        token: {
                            type: "string",
                            description: "The password reset token",
                        },
                        newPassword: {
                            type: "string",
                            description: "The new password",
                        },
                    },
                    required: ["token", "newPassword"],
                },
            },
        },
    },
    responses: {
        200: {
            description: "Password reset successfully, new password set",
            content: {
                "application/json": {
                    schema: {
                        type: "object",
                        properties: {
                            message: {
                                type: "string",
                                description: "Success message",
                            },
                            cookies: {
                                type: "object",
                                properties: {
                                    accessToken: {
                                        type: "string",
                                        description: "The new access token",
                                    },
                                    refreshToken: {
                                        type: "string",
                                        description: "The new refresh token",
                                    },
                                    sessionId: {
                                        type: "string",
                                        description: "The new session ID",
                                    },
                                    csrfToken: {
                                        type: "string",
                                        description: "The new CSRF token",
                                    },
                                },
                                required: ["accessToken", "refreshToken", "csrfToken"],
                            },
                        },
                    },
                },
            },
        },
        400: {
            description: "Invalid request (e.g., missing token or newPassword)",
        },
        401: {
            description: "Unauthorized or invalid token",
        },
    },
};
exports.default = async (data) => {
    const { body, ctx } = data;
    const { token, newPassword } = body;
    try {
        ctx === null || ctx === void 0 ? void 0 : ctx.step("Validating password reset token");
        if (!token || !newPassword) {
            ctx === null || ctx === void 0 ? void 0 : ctx.fail("Token and new password are required");
            throw (0, error_1.createError)({
                statusCode: 400,
                message: "Token and new password are required",
            });
        }
        ctx === null || ctx === void 0 ? void 0 : ctx.step("Verifying reset token");
        const decodedToken = await (0, token_1.verifyResetToken)(token);
        if (!decodedToken) {
            ctx === null || ctx === void 0 ? void 0 : ctx.fail("Invalid or expired token");
            throw new Error("Invalid token");
        }
        ctx === null || ctx === void 0 ? void 0 : ctx.step("Checking token usage");
        try {
            if (decodedToken.jti !== (await (0, utils_1.addOneTimeToken)(decodedToken.jti, new Date()))) {
                ctx === null || ctx === void 0 ? void 0 : ctx.fail("Token already used");
                throw (0, error_1.createError)({
                    statusCode: 500,
                    message: "Token has already been used",
                });
            }
        }
        catch (error) {
            ctx === null || ctx === void 0 ? void 0 : ctx.fail("Token validation failed");
            throw (0, error_1.createError)({
                statusCode: 500,
                message: error.message,
            });
        }
        ctx === null || ctx === void 0 ? void 0 : ctx.step("Hashing new password");
        const errorOrHashedPassword = await (0, passwords_1.hashPassword)(newPassword);
        const hashedPassword = errorOrHashedPassword;
        ctx === null || ctx === void 0 ? void 0 : ctx.step("Looking up user");
        const user = await db_1.models.user.findByPk(decodedToken.sub.user.id);
        if (!user) {
            ctx === null || ctx === void 0 ? void 0 : ctx.fail("User not found");
            throw (0, error_1.createError)({
                statusCode: 404,
                message: "User not found",
            });
        }
        ctx === null || ctx === void 0 ? void 0 : ctx.step("Updating password");
        await user.update({ password: hashedPassword });
        ctx === null || ctx === void 0 ? void 0 : ctx.step("Generating session tokens");
        const result = await (0, utils_1.returnUserWithTokens)({
            user,
            message: "Password reset successfully",
        });
        ctx === null || ctx === void 0 ? void 0 : ctx.success(`Password reset successfully for user ${user.email}`);
        return result;
    }
    catch (error) {
        ctx === null || ctx === void 0 ? void 0 : ctx.fail(error.message || "Password reset failed");
        throw error;
    }
};
